/*
 * Semi-generic scheduler for Fabric manager
 */

#include <stdio.h>
#ifdef _WIN32
#include "lf_win.h"
#else
#include <stdint.h>
#endif
#include <stdlib.h>
#ifndef _WIN32
#include <pthread.h>
#include <sys/time.h>
#endif

#include "libfma.h"
#include "lf_internal.h"
#include "lf_scheduler.h"

/* anchor of event list */
static struct lf_event Events = {
  NULL, NULL, {0, 0}, &Events, &Events, &Events, &Events
};
static int Event_breakout;

/*
 * Mutex for scheduler
 */
static pthread_mutex_t _lf_sched_lock;

/*
 * Initialize scheduler variables
 */
void
lf_scheduler_init()
{
  pthread_mutex_init(&_lf_sched_lock, NULL);
}

/*
 * Finalize scheduler variables
 */
void
lf_scheduler_finalize()
{
  pthread_mutex_destroy(&_lf_sched_lock);
}

/*
 * lf_schedule_event - add an event to the queue
 */
struct lf_event *
lf_schedule_event(
  void (*handler)(void *),
  void *context,
  uint32_t millisecs)
{
  struct lf_event *ep;
  struct lf_event *pep;
  int diff;
  int rc;

  ep = NULL;

#ifdef LF_TIMEWARP
  /* Convert timeouts from base-10 orders of magnitude to base-2.
   * (minimum timeout is 5s)
   */
  {
    unsigned long tens;
    unsigned long twos;

    tens=500;
    twos=5000;
    while (tens<millisecs) {
      tens += tens>>1;
      twos += twos>>3;
    }
    millisecs = twos;
  }
#endif

  /* allocate a struct for this event */
  LF_CALLOC(ep, struct lf_event, 1);

  /* fill it in with the time to fire */
  rc = gettimeofday(&ep->timeout, NULL);
  if (rc == -1) LF_ERROR(("gettimeofday failed"));
  
  tv_add(&ep->timeout, millisecs);

  /* fill in the rest */
  ep->handler = handler;
  ep->context = context;

  /* find place in list */
  pthread_mutex_lock(&_lf_sched_lock);
  pep = Events.prev;
  while (pep != &Events && (diff=tv_diff(&pep->timeout, &ep->timeout)) > 0) {
    pep = pep->prev;
  }

  /* If this event has the same timeout as pep, link into pep's peer list */
  if (pep != &Events && diff == 0) {
    ep->peer_prev = pep->peer_prev;
    ep->peer_next = pep;
    pep->peer_prev = ep;
    ep->peer_prev->peer_next = ep;

    ep->next = ep;
    ep->prev = ep;

  } else {
    /* place the new event after pep */
    ep->next = pep->next;
    ep->prev = pep;
    ep->next->prev = ep;
    ep->prev->next = ep;

    ep->peer_next = ep;
    ep->peer_prev = ep;
  }

  pthread_mutex_unlock(&_lf_sched_lock);

  /* return handle for the event */
  return ep;

 except:
  LF_FREE(ep);
  return NULL;
}

/*
 * remove and free and event from the queue
 */
void
lf_remove_event(
  struct lf_event *ep)
{
  /* just yank it out of the list and free it */
  if (ep != NULL) {

    /* protect for list access */
    pthread_mutex_lock(&_lf_sched_lock);

    /* If main next is someone else, we are a primary on main list */
    if (ep->next != ep) {

      /* if peer list is empty, simply unlink this element */
      if (ep->peer_next == ep) {

	/* unlink from list */
	ep->prev->next = ep->next;
	ep->next->prev = ep->prev;

      /* need to promote a peer to main list */
      } else {
	struct lf_event *nep;

	/* replace self with nep on main list */
	nep = ep->peer_next;
	ep->prev->next = nep;
	ep->next->prev = nep;
	nep->next = ep->next;
	nep->prev = ep->prev;

	/* link self out of peer list */
	ep->peer_prev->peer_next = ep->peer_next;
	ep->peer_next->peer_prev = ep->peer_prev;
      }
    
    /* only on peer list, just unlink from peer list */
    } else {
      ep->peer_prev->peer_next = ep->peer_next;
      ep->peer_next->peer_prev = ep->peer_prev;
    }

    pthread_mutex_unlock(&_lf_sched_lock);

    LF_FREE(ep);
  }
}

/*
 * run all event handlers that are ready
 */
int
lf_check_events()
{
  struct lf_event *ep;
  struct lf_event *fep;
  struct timeval now;
  int rc;
  int diff;

  rc = gettimeofday(&now, NULL);
  if (rc == -1) return -1;

  pthread_mutex_lock(&_lf_sched_lock);
  ep = Events.next;

  while (ep != &Events && tv_diff(&ep->timeout, &now) <= 0) {

    /* unlink now to make sure the list remains consistant */
    ep->prev->next = ep->next;
    ep->next->prev = ep->prev;

    /* Must unlock before calling handler to avoid deadlock */
    pthread_mutex_unlock(&_lf_sched_lock);

    /* run handler for each event on peer list */
    fep = ep;
    do {
      struct lf_event *nep;

      ep->handler(ep->context);	/* run the handler */

      nep = ep->peer_next;	/* save next pointer */
      LF_FREE(ep);		/* release this event struct */

      ep = nep;			/* next in list */
    } while (ep != fep);

    pthread_mutex_lock(&_lf_sched_lock);	/* re-lock */

    /*
     * If breakout requested by a handler, stop processing events
     * and allow something else to run.
     */
    if (Event_breakout) {
      Event_breakout = FALSE;
      break;
    }

    ep = Events.next;			/* advance to next event */
  }

  /* return ms remaining until next event, or 0 if none */
  if (Events.next == &Events) {
    diff = -1;
  } else {

    /* get time all over again, since the handlers may have taken a while */
    rc = gettimeofday(&now, NULL);
    if (rc == -1) {
      diff = 0;
    } else {
      diff =  tv_diff(&Events.next->timeout, &now);
      if (diff < 0) diff = 0;
    }
  }

  pthread_mutex_unlock(&_lf_sched_lock);

  return diff;
}

/*
 * Request a breakout of the scheduler - this is called by a handler that
 * wants to allow something else to run, and then run a scheduled event
 * as quickly as possible, e.g.:
 * some_handler()
 * {
 *   lf_schedule_event(some_rtn, arg, 0);
 *   lf_schedule_breakout();
 * }
 * Then, when some_handler() returns, lf_check_events() will return
 * after finishing peers of some_handler() but before running anything
 * else on the list, even though time is up, such as some_rtn()
 */
void
lf_schedule_breakout()
{
  Event_breakout = TRUE;
}
